/**
 * The report type that gets passed to Watson
 *
 * @export
 * @enum {number}
 */
export declare enum WerReportType {
    /**
     * will send silently if sufficient consent is given.
     */
    NonCritical = 0,
    /**
     * Will show a crash dialog
     */
    Critical = 1,
    AppCrash = 2,
    AppHang = 3,
    Invalid = 5,
}
/**
 * Tells Watson how to include a file in a report.
 *
 * @export
 * @interface WerFileInfo
 */
export interface WerFileInfo {
    /**
     * The absolute path to the file
     *
     * @type {string}
     * @memberOf WerFileInfo
     */
    path: string;
    /**
     * The file type parameter passed to Watson (see WerFileType)
     *
     * @type {WerFileType}
     * @memberOf WerFileInfo
     */
    fileType: WerFileType;
    anonymousData: boolean;
    deleteWhenDone: boolean;
}
/**
 * The valid file types
 *
 * @export
 * @enum {number}
 */
export declare enum WerFileType {
    /**
     * An arbitrary file with information about the error
     */
    Other = 5,
    /**
     * Something user created that may have had something to do with the crash
     */
    UserDoc = 4,
}
/**
 * A bucket parameter passed to Watson
 *
 * @export
 * @interface WerParam
 */
export interface WerParam {
    /**
     * The parameter number. 0-9 are the only valid options.
     *
     * @type {number}
     * @memberOf WerParam
     */
    paramNumber: number;
    /**
     * The paramter value
     *
     * @type {string}
     * @memberOf WerParam
     */
    paramValue: string;
    /**
     * The paramter name
     *
     * @type {string}
     * @memberOf WerParam
     */
    paramName?: string;
}
/**
 * The consent passed to Watson
 *
 * @export
 * @enum {number}
 */
export declare enum WerConsent {
    /**
     * This indicates you have not previously asked for consent, so Watson may ask in its dialogs before sending PII.
     */
    WerConsentNotAsked = 1,
    /**
     * This indicates the user has previously consented to send PII in this instance.
     */
    WerConsentApproved = 2,
    /**
     * The user has previously denied consent to send PII
     */
    WerConsentDenied = 3,
    /**
     * Watson should prompt the user regardless of prior consent
     */
    WerConsentAlwaysPrompt = 4,
    /**
     * COnsent has been given to collect as much information as possible
     */
    WerConsentMax = 5,
}
/**
 * Flags that can be added to a report to change the behavior.
 * Add all the flags you want to include together
 *
 * @export
 * @enum {number}
 */
export declare enum WerSubmitFlags {
    /**
     * show recovery option
     */
    WER_SUBMIT_HONOR_RECOVERY = 1,
    /**
     * show application restart option
     */
    WER_SUBMIT_HONOR_RESTART = 2,
    /**
     * report directly to queue
     */
    WER_SUBMIT_QUEUE = 4,
    /**
     * show the debug button
     */
    WER_SUBMIT_SHOW_DEBUG = 8,
    /**
     * Add registered data to the WER report
     */
    WER_SUBMIT_ADD_REGISTERED_DATA = 16,
    /**
     * Force the report to go out of process
     */
    WER_SUBMIT_OUTOFPROCESS = 32,
    /**
     * Do not show the close dialog for the critical report
     */
    WER_SUBMIT_NO_CLOSE_UI = 64,
    /**
     * Do not queue the report
     */
    WER_SUBMIT_NO_QUEUE = 128,
    /**
     * Do not archive the report
     */
    WER_SUBMIT_NO_ARCHIVE = 256,
    /**
     * The initial reporting UI is minimized and will flash
     */
    WER_SUBMIT_START_MINIMIZED = 512,
    /**
     * Force the report to go out of process and do not wait for it to finish
     */
    WER_SUBMIT_OUTOFPROCESS_ASYNC = 1024,
    /**
     * Bypass data throttling for the report
     */
    WER_SUBMIT_BYPASS_DATA_THROTTLING = 2048,
    /**
     * Archive only the parameters, the cab is discarded
     */
    WER_SUBMIT_ARCHIVE_PARAMETERS_ONLY = 4096,
    /**
     * Always send the machine ID, regardless of the consent the report was submitted with
     */
    WER_SUBMIT_REPORT_MACHINE_ID = 8192,
    /**
     * Bypass power-related throttling (when on battery)
     */
    WER_SUBMIT_BYPASS_POWER_THROTTLING = 16384,
    /**
     * Bypass network-related throttling (when on restricted networks)
     */
    WER_SUBMIT_BYPASS_NETWORK_COST_THROTTLING = 32768,
}
/**
 * Creates and sends a watson report
 *
 * @export
 * @param {string} eventType
 * @param {WerReportType} reportType
 * @param {WerConsent} consent
 * @param {number} submitFlags
 * @param {WerFileInfo[]} [files]
 * @param {WerParam[]} [params]
 */
export declare function CreateWerReport(eventType: string, reportType: WerReportType, consent: WerConsent, submitFlags: number, files?: WerFileInfo[], params?: WerParam[]): void;
